<?php
namespace App\Http\Controllers;

use App\LedgerSubcategory;
use App\LedgerType;
use Illuminate\Http\Request;

class LedgerSubcategoryController extends Controller
{
    public function index()
    {
        // eager load ledgerType
        $subcategories = LedgerSubcategory::with('ledgerType')->orderBy('created_at','desc')->paginate(20);
        return view('ledger_subcategories.index', compact('subcategories'));
    }

    public function create()
    {
        $ledgerTypes = LedgerType::orderBy('name')->get(['id','name']);
        return view('ledger_subcategories.create', compact('ledgerTypes'));
    }

    public function store(Request $request)
    {
        // dd($request->all());
        $data = $request->validate([
            'ledger_type_id' => 'required|exists:ledger_types,id',
            'name'           => 'required|string|max:150',
            'description'    => 'nullable|string',
            'is_active'      => 'nullable'
        ]);
        // dd('adsda');

        // ensure unique per type
        $exists = LedgerSubcategory::where('ledger_type_id', $data['ledger_type_id'])
            ->where('name', $data['name'])
            ->exists();

        if ($exists) {
            return back()->withInput()->withErrors([
                'name' => 'This subcategory already exists for the selected type.'
            ]);
        }

        // convert checkbox to 1/0
        $data['is_active'] = $request->has('is_active') ? 1 : 0;

        // save
        $subcategory = LedgerSubcategory::create($data);

        return redirect()
            ->route('ledger-subcategories.index')
            ->with('success', 'Subcategory created: '.$subcategory->name);
    }

    public function edit($id)
    {
        $sub = LedgerSubcategory::findOrFail($id);
        $ledgerTypes = LedgerType::orderBy('name')->get(['id','name']);
        return view('ledger_subcategories.edit', compact('sub','ledgerTypes'));
    }

    public function update(Request $request, $id)
    {
        $sub = LedgerSubcategory::findOrFail($id);

        $data = $request->validate([
            'ledger_type_id' => 'required|exists:ledger_types,id',
            'name' => 'required|string|max:150',
            'description' => 'nullable|string',
            'is_active' => 'nullable|boolean'
        ]);

        $exists = LedgerSubcategory::where('ledger_type_id', $data['ledger_type_id'])
            ->where('name', $data['name'])
            ->where('id', '<>', $id)
            ->exists();

        if ($exists) {
            return back()->withInput()->withErrors(['name' => 'This subcategory already exists for the selected type.']);
        }

        $data['is_active'] = $request->has('is_active') ? 1 : 0;
        $sub->update($data);

        return redirect()->route('ledger-subcategories.index')->with('success','Subcategory updated.');
    }

    public function destroy($id)
    {
        $sub = LedgerSubcategory::findOrFail($id);
        $sub->delete();
        return back()->with('success','Subcategory deleted.');
    }

    public function byType($typeId)
    {   
        return LedgerSubcategory::where('ledger_type_id', $typeId)
            ->where('is_active', 1)
            ->select('id','name')
            ->get();
    }
}
