<?php

namespace App\Http\Controllers;

use App\BusinessPartner;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class BusinessPartnerController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            // Get all partners without filtering by business_id
            $partners = BusinessPartner::query();

            return DataTables::of($partners)
                ->addColumn('action', function ($row) {
                    return '
                        <button data-href="'.action([BusinessPartnerController::class, 'edit'], [$row->id]).'" 
                                class="btn btn-xs btn-primary btn-modal" data-container=".partner_edit_modal">
                            <i class="glyphicon glyphicon-edit"></i> Edit
                        </button>
                        <button data-href="'.action([BusinessPartnerController::class, 'destroy'], [$row->id]).'" 
                                class="btn btn-xs btn-danger delete_partner_button">
                            <i class="glyphicon glyphicon-trash"></i> Delete
                        </button>';
                })
                ->editColumn('phone', fn($row) => $row->phone ?? '-')
                ->editColumn('email', fn($row) => $row->email ?? '-')
                ->editColumn('address', fn($row) => $row->address ?? '-')
                ->rawColumns(['action'])
                ->make(true);
        }

        return view('business_partners.index');
    }


    public function create()
    {
        return view('business_partners.create');
    }

    public function store(Request $request)
    {
        //$business_id = $request->session()->get('user.business_id');

        try {

            $data = $request->validate([
                'name' => 'required|string|max:255',
                'phone' => 'nullable|string|max:255',
                'email' => 'nullable|email|max:255',
                'address' => 'nullable|string|max:500',
                'percentage' => 'required|numeric|min:0|max:100',
                'details' => 'nullable|string',
            ]);

            //$data['business_id'] = $business_id;

            BusinessPartner::create($data);

            return redirect()
                ->route('business-partners.index')   // 👈 redirect to index.blade.php
                ->with('success', __('Partner Details Added successfully'));

            // $output = ['success' => true,
            //     'msg' => __('business.business_location_added_success'),
            // ];
            // return $output;
            // return ['success' => true, 'msg' => __('Partner added successfully')];

        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            return redirect()
                ->back()
                ->with('error', 'Something went wrong!');
        }

    }

    public function edit($id)
    {
        // dd($id);

        $partner = BusinessPartner::findOrFail($id);

        return view('business_partners.edit')->with(compact('partner'));
    }

    public function update(Request $request, $id)
    {
        //$business_id = $request->session()->get('user.business_id');

        try {

            $data = $request->validate([
                'name' => 'required|string|max:255',
                'phone' => 'nullable|string|max:255',
                'email' => 'nullable|email|max:255',
                'address' => 'nullable|string|max:500',
                'percentage' => 'required|numeric|min:10|max:90', // your earlier rule
                'status' => 'required|in:Active,Inactive',        // 👈 add this
                'details' => 'nullable|string',
            ]);

            // dd($data, $request->all());
            BusinessPartner::where('id', $id)
                ->update($data);

            return redirect()
                ->route('business-partners.index')   // 👈 redirect to index.blade.php
                ->with('success', 'Partner Details Updated Successfully');

        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            return redirect()
                ->back()
                ->with('error', 'Something went wrong!');
        }

        // return ['success' => true, 'msg' => __('Partner updated successfully')];
    }

    public function destroy($id)
    {
        try {
            BusinessPartner::where('id', $id)->delete();

            return response()->json([
                'success' => true,
                'msg' => 'Partner deleted successfully!'
            ]);
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().' Line:'.$e->getLine().' Message:'.$e->getMessage());

            return response()->json([
                'success' => false,
                'msg' => 'Something went wrong!'
            ], 500);
        }
    }

}
